
/* Copyright (C) 2001-2010 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_xmalloc.c */


#include "fs_itype.h"
#include "fs_bitmap.h"
#include "fs_graymap.h"

/* local prototypes */

#if !defined(FS_INT_MEM)
static FS_VOID * fs__malloc(_DS_ FS_ULONG bytes);
static FS_LONG fs__free(_DS_ FS_VOID *ptr);
static FS_VOID * fs__realloc(_DS_ FS_VOID *ptr, FS_ULONG n);
#endif

/* not used
#ifdef FS_DEBUG
#define FS_DEBUG_TAG (FS_ULONG)0x91827364
#endif*/

/* internal memory case is implemented in fs_bestfit.c */

#ifndef FS_INT_MEM

#if defined(FS_FREE_SIZE_DEFINED)
/****************************************************************/
/***** operating systems MALLOC/FREE with FS_FREE_SIZE() ********/
/****************************************************************/

/* This layer maintains the 'highwater mark' variable <allocated>
* when the operating system provides a call to inquire about the size
* of an object on the heap (about to be free-d). Makes this easy.
*/

/****************************************************************/
static FS_VOID * fs__malloc(_DS_ FS_ULONG n)
{
    FS_VOID * p;

    if ((STATE.server->allocated + n) > STATE.server->heap_size)
        return 0;

    p = SYS_MALLOC(n);
    if (p)
        STATE.server->allocated += FS_FREE_SIZE(p);
    return p;
}

/****************************************************************/
static FS_LONG fs__free(_DS_ FS_VOID *ptr)
{
    STATE.server->allocated -= FS_FREE_SIZE(ptr);
    SYS_FREE(ptr);
    return SUCCESS; /* !!! actually no way of knowing !!! */
}

/****************************************************************/
static FS_VOID * fs__realloc(_DS_ FS_VOID *ptr, FS_ULONG n)
{
    FS_ULONG old;

    old = FS_FREE_SIZE(ptr);

    if ((STATE.server->allocated + n - old) > STATE.server->heap_size)
        return 0;

    ptr = SYS_REALLOC(ptr, n);
    if (ptr)
    {
        STATE.server->allocated -= old;
        STATE.server->allocated += FS_FREE_SIZE(ptr);
    }
    return ptr;
}

#else
/****************************************************************/
/************* stone axes and animal skins **********************/
/****************************************************************/

/* This layer maintains the 'highwater mark' variable <allocated>
* when the OS doesn't have a call to tell us the size of a pointer.
* So we need to allocate some extra space to hold the size field.
* But can't just be a FS_ULONG, since this screws up alignments on
* machines like the ALPHA which want pointers to be on 8 byte
* boundaries.  So we use a union -- let the compiler do the math!
*/
typedef union
{
    FS_ULONG *ptr;    /* just for alignment purposes */
    FS_ULONG size;
} SIZE_THING;

/****************************************************************/
static FS_VOID * fs__malloc(_DS_ FS_ULONG n)
{
    SIZE_THING *size;

    n += sizeof(SIZE_THING);

    if (STATE.server->allocated + n > STATE.server->heap_size)
        return 0;

    size = (SIZE_THING *)SYS_MALLOC(n);
    if (size == 0)
        return 0;

    size->size = n;
    STATE.server->allocated += n;
    size++;        /* skip over header */
    return (FS_VOID *) size;
}

/****************************************************************/
static FS_LONG fs__free(_DS_ FS_VOID *ptr)
{
    SIZE_THING *size;

    size = (SIZE_THING *)ptr;
    size--;    /* now pointing at the header */

    STATE.server->allocated -= size->size;
    SYS_FREE((FS_VOID *)size);
    return SUCCESS;    /* !!! actually no way of knowing !!! */
}

/****************************************************************/
static FS_VOID * fs__realloc(_DS_ FS_VOID *ptr, FS_ULONG n)
{
    SIZE_THING *size;
    FS_LONG old_size;

    size = (SIZE_THING *)ptr;
    size--;    /* at old header */
    old_size = size->size;

    n += sizeof(SIZE_THING);

    if (STATE.server->allocated + n - old_size > STATE.server->heap_size)
        return 0;

    size = (SIZE_THING *) SYS_REALLOC((FS_VOID *)size, n);

    if (size)
    {
        size->size = n;
        STATE.server->allocated -= old_size;
        STATE.server->allocated += n;
        size++;    /* skip the header */
    }
    return (FS_VOID *) size;
}

#endif

#endif /* !FS_INT_MEM */

/****************************************************************/
/***************** new function to trim the cache ***************/
/****************************************************************/
#ifdef FS_CACHE

static int trim_cache(_DS0_)
{
    int i, got_some = 0;
    CACHE_ENTRY **cache, *p, *head, *lru;

    /* kill the LRU character with no references from each hash chain */
    cache = STATE.server->cache;
    for (i = 0; i < CACHE_MOD; i++)
    {
        head = (CACHE_ENTRY *)(cache[i]);
        if (head)
        {
            p = lru = (CACHE_ENTRY *)(head->prev);

            /* not if there are outstanding references */
            while (p->ref_counter && (p != head))
            {
                p = (CACHE_ENTRY *)(p->prev);
            }
            if (p->ref_counter)
                continue;

            /* unlink p */

            /* ? singleton ... kill the entire chain */
            if ((CACHE_ENTRY *)(p->prev) == p)
            {
                cache[i] = 0;
            }
            else if (p == head)
            {
                cache[i] = p->next;
                head = (CACHE_ENTRY *)(cache[i]);
                head->prev = p->prev;
            }
            else if (p == lru)
            {
                head->prev = p->prev;
                p->prev->next = 0;
            }
            else
            {
                p->next->prev = p->prev;
                p->prev->next = p->next;
            }

            /* update the SFNT's count of cached characters */
            if (p->sfnt->cache_count)
            {
                p->sfnt->cache_count--;
            }

            /* now really free the character and the container */
            FSS_free(_PS_ (FS_BYTE *)(p->data));
#ifdef FS_NO_FS_LUMP
            FSS_free(_PS_ p);
#else
            CACHE_ENTRY_free(_PS_ p);
#endif
            got_some++;
        }
    }
    return got_some;
}

#else /* there is no cache to trim, we got nothing back */

static int trim_cache(_DS0_)
{
    FS_state_ptr = FS_state_ptr;    /* rid of warning */
    return 0;
}

#endif /* FS_CACHE */

static int kill_unused_sfnts(_DS0_)
{
    SFNT *prev, *sfnt, *next;
    SENV *senv;
    int got_some = 0;

    prev = next = NULL;

    for (sfnt = (SFNT *)(STATE.server->scaled_fonts);
         sfnt;
         prev = sfnt, sfnt = (SFNT *)(next))
    {
        next = sfnt->next;

        /* never any client's current font */
        if (sfnt->ref_count)
        {
            continue;
        }
        /* nor any font with active characters */
        if (sfnt->active_count)
        {
            continue;
        }
        /* no characters in cache at all -- kill it */
        if (sfnt->cache_count == 0)
        {
#ifdef FS_DEBUG
            /*** check the cache to verify count ***/
            {
                CACHE_ENTRY * cp, **cache = STATE.server->cache;
                int i, num = 0;

                for (i = 0; i < CACHE_MOD; i++)
                {
                    for (cp = (CACHE_ENTRY *)(cache[i]); cp; cp = (CACHE_ENTRY *)(cp->next))
                    {
                        if (((SFNT *)(cp->sfnt) == sfnt) &&
                            (cp->ref_counter != 0))
                        {
                            num++;
                        }
                    }
                }
                if (num != 0)
                {
                    FS_PRINTF(("*** oops used cache count for %p should be ZERO, is %d\n", (void *)sfnt, num));
                    STATE.error = ERR_FONT_IN_USE;
                    return 0;
                }
            }
#endif /* FS_DEBUG */

#ifdef FS_CACHE
            delete_fs_cache(_PS_ sfnt);
#endif
            /* change the head? */
            if ((SFNT *)(STATE.server->scaled_fonts) == sfnt)
            {
                STATE.server->scaled_fonts = sfnt->next;
            }
            else
            {
                /* unlink it */
                if (prev)
                    prev->next = next;
            }
            /* kill sfnt - but don't free the LFNT */
            senv = (SENV *)(sfnt->senv);
            if (senv)
            {
                if ((sfnt->lfnt->fnt_type == TTF_TYPE) ||
                    (sfnt->lfnt->fnt_type == TTC_TYPE) ||
                    (sfnt->lfnt->fnt_type == CFF_TYPE) )
                    delete_key(_PS_ (FS_VOID *)(sfnt->senv->ttkey));
                else
                    FSS_free(_PS_ (FS_VOID *)(sfnt->senv->ttkey));

                FSS_free(_PS_ (SENV *)(sfnt->senv));
                senv = 0;
            }
            FSS_free(_PS_ sfnt);

            sfnt = prev; /* so that "prev" doesn't change */

            got_some++;
        }
    }
    return got_some;
}

/* unload the SENV for all SFNTs that are not the current STATE's current SFNT */
/* other clients will have to recreate their current SFNTs SENV before using   */
static int gut_other_sfnts(_DS0_, FS_BOOLEAN saveCurrentSFNTs)
{
    SFNT *sfnt;
    SENV *senv;
    int got_some = 0;

    for (sfnt = (SFNT *)(STATE.server->scaled_fonts);
         sfnt;
         sfnt = (SFNT *)(sfnt->next))
    {
        /* never the current client's current font */
        if (sfnt == STATE.cur_sfnt)
        {
            continue;
        }

        /* never mess with any process' current SFNT */
        if (saveCurrentSFNTs && (sfnt->ref_count > 0)) continue;

        /* ok ... we're free to gut the SFNT */

        /* kill SENV - but don't free the LFNT or SFNT */
        senv = (SENV *)(sfnt->senv);
        if (senv)
        {
            if ((sfnt->lfnt->fnt_type == TTF_TYPE) ||
                (sfnt->lfnt->fnt_type == TTC_TYPE) ||
                (sfnt->lfnt->fnt_type == CFF_TYPE) )
                delete_key(_PS_ (FS_VOID *)(senv->ttkey));
            else
                FSS_free(_PS_   (FS_VOID *)(senv->ttkey));

            FSS_free(_PS_ senv);
            sfnt->senv = 0;
            got_some++;
        }
    }
    return got_some;
}

/****************************************************************/
FS_VOID delete_lfnt(_DS_ LFNT *lfnt)
{
    SFNT *sfnt;

    /* multiple clients using the same name do not delete any lfnt */

    /* unlink the LFNT */
    if ((LFNT *)(STATE.server->loaded_fonts) == lfnt)
    {
        STATE.server->loaded_fonts = lfnt->next;
    }
    else
    {
        LFNT *prev;

        for (prev = (LFNT *)(STATE.server->loaded_fonts);
             (LFNT *)(prev->next) != lfnt;
             prev = (LFNT *)(prev->next))
        {
            ; /* do nothing */
        }
        prev->next = lfnt->next;
    }

    /* gut and delete the LFNT */
    delete_kern(_PS_ lfnt);

    unload_fnt(_PS_ lfnt);

    FSS_free(_PS_ (FILECHAR *)(lfnt->name));
    if (lfnt->path) FSS_free(_PS_ (FILECHAR *)(lfnt->path));

    for (sfnt = (SFNT *)(STATE.server->scaled_fonts); sfnt; sfnt = (SFNT *)(sfnt->next))
    {
        if ((LFNT *)(sfnt->lfnt) == lfnt)
        {
            sfnt->lfnt = 0;
        }
    }
#ifdef FS_CACHE_CMAP
    if (lfnt->cmap_cache) FSS_free(_PS_ (FS_CMAP_CACHE *)(lfnt->cmap_cache));
#endif
    FSS_free(_PS_ lfnt);
}

/****************************************************************/
FS_VOID kill_fntset(_DS_ FNTSET *fntset)
{
    CFNT *cfnt;
    FS_USHORT nc;

    /* we are going to kill this fontset, so decrement the fntset_refs
       count for each of the loaded fonts we reference
    */
    for (cfnt = (CFNT *)fntset->cfnt, nc = 0; nc < fntset->num_fonts; cfnt++, nc++)
    {
        LFNT *lfnt;

        lfnt = (LFNT *)(cfnt->lfnt);

        lfnt->fntset_refs--;

        if (lfnt->fntset_refs < 1)
        {
            SFNT *prev, *sfnt, *next;

            /* unlink/delete the SFNTs based on this LFNT */

            prev = next = NULL;

            for (sfnt = (SFNT *)(STATE.server->scaled_fonts);
                    sfnt;
                    prev = sfnt, sfnt = next)
            {
                next = (SFNT *)(sfnt->next);

                if ((LFNT *)(sfnt->lfnt) == lfnt)
                {
                    SENV *senv;

                    /* ? update head */
                    if ((SFNT *)(STATE.server->scaled_fonts) == sfnt)
                    {
                        STATE.server->scaled_fonts = sfnt->next;
                    }
                    else
                    {
                        /* remove <sfnt> from list */
                        if (prev)
                            prev->next = sfnt->next;
                    }
#ifdef FS_CACHE
                    /* kill all the cached_chars - all have a ref_count of 0 */
                    delete_fs_cache(_PS_ sfnt);
#endif
                    /* kill the senv if it exists - may not have been created or may be gutted */
                    senv = (SENV *)(sfnt->senv);
                    if (senv)
                    {
                        if ((sfnt->lfnt->fnt_type == TTF_TYPE) ||
                            (sfnt->lfnt->fnt_type == TTC_TYPE) ||
                            (sfnt->lfnt->fnt_type == CFF_TYPE) )
                            delete_key(_PS_ (FS_VOID *)(sfnt->senv->ttkey));
                        else
                            FSS_free(_PS_ (FS_VOID *)(sfnt->senv->ttkey));

                        FSS_free(_PS_ (SENV *)(sfnt->senv));
                    }
                    FSS_free(_PS_ sfnt);

                    sfnt = prev; /* so that "prev" doesn't change */
                }
            }
            delete_lfnt(_PS_ lfnt);
        }
        cfnt->name = 0;
    }
    FSS_free(_PS_ (CFNT *)(fntset->cfnt));
    FSS_free(_PS_ (FILECHAR *)(fntset->name));
    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));

    if ((FNTSET_OT_TABLE *)(fntset->cmap))
        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
    if ((FNTSET_OT_TABLE *)(fntset->gdef))
        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
    if ((FNTSET_OT_TABLE *)(fntset->gsub))
        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
    if ((FNTSET_OT_TABLE *)(fntset->gpos))
        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));

    if ((FNTSET *)(STATE.server->font_sets) == fntset)
    {
        STATE.server->font_sets = fntset->next;
    }
    else
    {
        FNTSET *prev;

        for (prev = (FNTSET *)(STATE.server->font_sets);
             (FNTSET *)(prev->next) != fntset;
             prev = (FNTSET *)(prev->next))
        {
            ; /* do nothing */
        }
        prev->next = fntset->next;
    }

    FSS_free(_PS_ fntset);
}

static int kill_tlists(_DS0_)
{
    int got_some = 0;
#if defined(FS_BITMAPS)
    if ((TLIST *)(STATE.server->tlist) || (TLIST *)(STATE.server->drop))
    {
        if (STATE.server->tlist && STATE.server->tlist->ref_count == 0)
        {
            delete_tlist(_PS_ (TLIST *)(STATE.server->tlist));
            STATE.server->tlist = 0;
            got_some++;
        }
        if (STATE.server->drop && STATE.server->drop->ref_count == 0)
        {
            delete_tlist(_PS_ (TLIST *)(STATE.server->drop));
            STATE.server->drop = 0;
            got_some++;
        }
    }
#else
    FS_state_ptr = FS_state_ptr;
#endif
    return got_some;
}

static int gut_raster_area(_DS0_)
{
    int got_some = 0;
#if defined(FS_GRAYMAPS)
    if (STATE.server->raster)
    {
        RASTER *r = (RASTER *)(STATE.server->raster);
        if (r->ref_count == 0)
        {
            FSS_free(_PS_ (FS_FIXED *)(r->areas));
            FSS_free(_PS_ r);
            STATE.server->raster = 0;
            got_some++;
        }
    }
#else
    FS_state_ptr = FS_state_ptr;
#endif
    return got_some;
}

static int gut_cmap_caches(_DS0_)
{
    int got_some = 0;
#if defined(FS_CACHE_CMAP)
    LFNT *lfnt;
    for (lfnt = (LFNT *)(STATE.server->loaded_fonts); lfnt; lfnt = (LFNT *)(lfnt->next))
    {
        if (lfnt->cmap_cache)
        {
            unload_cmap_cache(_PS_ lfnt);
            got_some++;
        }
    }
#else
    FS_state_ptr = FS_state_ptr; /* avoid compiler warning */
#endif
    return got_some;
}

static int kill_unused_fntsets(_DS0_)
{
    FNTSET *fntset;
    int got_some = 0;

    for (fntset = (FNTSET *)(STATE.server->font_sets); fntset; fntset = (FNTSET *)(fntset->next))
    {
        /* is this fontset a current fontset of any client/STATE ?
         */
        if (fntset->ref_count > 0) /* yes */ continue;

        if (fntset->count_added == 0)
        {
            kill_fntset(_PS_ fntset);

            got_some = 1;

            break;
        }
    }
    return got_some;
}

/****************************************************************/
/* a new function to gut unused FNTSET's */
static int gut_unused_fntsets(_DS0_, FS_BOOLEAN saveCurrentFntsets)
{
    FNTSET *fntset;
    int got_some = 0;

    for (fntset = (FNTSET *)(STATE.server->font_sets); fntset; fntset = (FNTSET *)(fntset->next))
    {
        CFNT *cfnt;
        FS_USHORT ncmp;

        got_some += squeeze_table_ptrs(_PS_ fntset);

        /* never mess with any process' current FNTSET */
        if (saveCurrentFntsets && (fntset->ref_count > 0)) continue;

        /* ok ... we're free to gut the FNTSET */

        for (cfnt = (CFNT *)fntset->cfnt, ncmp = 0; ncmp < fntset->num_fonts; cfnt++, ncmp++)
        {
            LFNT *lfnt;
            FNTSET *fs;
            FS_BOOLEAN do_continue;

            lfnt = (LFNT *)(cfnt->lfnt);

            /*
                I don't like the following check, but this case happens
                (lfnt is NULL) when FSS_add_font_with_offset() hasn't
                yet allocated and assigned the lfnt, but is still trying
                to get a unique_font_name().  The load_fnt() that's called
                from unique_font_name() calls get_some_back(), yada-yada
            */
            if (!lfnt) continue;

            if (any_used_lfnt_table_ptrs(_PS_ fntset, lfnt)) continue;

            /*
                What about the case where this lfnt is linked by a
                different fntset (shared component), and the other fntset
                is someone's current FNTSET.  That is, this lfnt may be
                a current LFNT -- a component of a current FNTSET.
            */
            if (saveCurrentFntsets)
            {
                do_continue = 0;
                for (fs = (FNTSET *)(STATE.server->font_sets); fs; fs = (FNTSET *)(fs->next))
                {
                    /* some process' current FNTSET */
                    if (fs->ref_count > 0)
                    {
                        FS_USHORT nc;
                        CFNT *cf;

                        for (nc = 0, cf = (CFNT *)(fs->cfnt); nc < fs->num_fonts; nc++, cf++)
                        {
                            LFNT *lf;

                            lf = (LFNT *)(cf->lfnt);

                            if (lf == lfnt)
                            {
                                do_continue = 1;
                                break;
                            }
                        }
                        if (do_continue) break;
                    }
                }
                if (do_continue) continue;
            }

            if (lfnt->fntset_refs)
            {
                /* unload ttf [and decomp] */
                if (!lfnt->loading && (lfnt != STATE.cur_lfnt) && (FS_VOID *)(lfnt->fnt))
                {
                    SFNT *sfnt;

                    unload_fnt(_PS_ lfnt);

                    /* maxp was nulled in unload_fnt, mark the corresponding
                       kep->maxp nullified too
                    */
                    for (sfnt = (SFNT *)(STATE.server->scaled_fonts); sfnt; sfnt = (SFNT *)(sfnt->next))
                    {
                        if (((LFNT *)(sfnt->lfnt) == lfnt) &&
                            (SENV *)(sfnt->senv) &&
                            sfnt->senv->ttkey)
                        {
                            ((fsg_SplineKey *)(sfnt->senv->ttkey))->maxp = 0;
                        }
                    }
                    got_some++;
                }
            }
            else
            {
                /* this should never happen !!!!! */
                /* if it does, something is wrong somewhere else */

                delete_lfnt(_PS_ lfnt);
                got_some++;
            }
        }
    }
    return got_some;
}

/****************************************************************/
/***************** exports **************************************/
/****************************************************************/
int get_some_back(_DS0_)
{
    int got_some;

    /* first remove any deleted, unreferenced, unused fntsets */
    got_some = kill_unused_fntsets(_PS0_);

    /* first remove the lru entry (if unused) from each cache line */
    if (!got_some) got_some += trim_cache(_PS0_);

    /* remove any SFNT's that have no characters in cache -- expensive to re-create */
    if (!got_some) got_some += kill_unused_sfnts(_PS0_);

    /* un-load any FNTSET unused in any SFNT's, and unused in <table_ptrs> */
    /* we might be re-loading a compressed font -- could get expensive */
    if (!got_some) got_some += gut_unused_fntsets(_PS0_, 1);

    /* unload the grayscale raster area within the server -- will be restored next time */
    /* but may reduce fragmentation -- kills performance if done too often */
    if (!got_some) got_some += gut_raster_area(_PS0_);

    /* remove TLIST if bitmaps were being rendered -- will be restored next time */
    /* but may reduce fragmentation -- kills performance if done too often       */
    if (!got_some) got_some += kill_tlists(_PS0_);

    /* remove cmap cache for LFNTs -- hurts performance */
    if (!got_some) got_some += gut_cmap_caches(_PS0_);

    /* un-load a FNTSET even if it's used in any SFNT's, and unused in <table_ptrs> */
    /* we might be re-loading a compressed font -- could get expensive */
    if (!got_some) got_some += gut_unused_fntsets(_PS0_, 0);

    /* unload the SENV of SFNT's that are not the current SFNT */
    /* of any client -- very expensive to re-create            */
    if (!got_some) got_some += gut_other_sfnts(_PS0_, 1);

    /* last resort - unload the SENV of SFNT's even if they are the current SFNT */
    /* of another client, but never gut the current STATE's current SFNT         */
    /* -- very very expensive                                                    */
    if (!got_some) got_some += gut_other_sfnts(_PS0_, 0);

    return got_some;
}

/****************************************************************/
/* try extra hard to allocate that buffer */
FS_VOID *FSS_malloc(_DS_ FS_ULONG n)
{
    FS_BYTE *p;
    int ok = 1;

    if (n >= STATE.server->heap_size)
    {
        STATE.error = ERR_MALLOC_FAIL;
        return 0;
    }
    if (n == 0)
    {
        return 0;
    }
    while (ok)
    {
        STATE.error = SUCCESS;
        p = (FS_BYTE *) fs__malloc(_PS_ n);
        if (p)
        {
#ifdef FS_MEM_DBG
            FS_FPRINTF((FS_state_ptr->memdbgfp, "MALLOC %p %u %s\n", p, n,
                        FS_state_ptr->memdbgid ? FS_state_ptr->memdbgid : "[NO_LABEL]"));
            FS_state_ptr->memdbgid = NULL;
            FS_FFLUSH(FS_state_ptr->memdbgfp);
#endif
            return (FS_VOID *)p;
        }
        ok = get_some_back(_PS0_);
#ifdef FS_DEBUG
        /*** for debugging set breakpoint on next get_some_back()    ***/
        /*** to see why get_some_back(), isn't getting anything back ***/
        if (!ok)
        {
            get_some_back(_PS0_);
        }
#endif
    }
    STATE.error = ERR_MALLOC_FAIL;
    return 0;
}

/****************************************************************/
/* initialize the buffer to 0's */
FS_VOID *FSS_calloc(_DS_ FS_ULONG n)
{
    FS_BYTE *p;

    p = FSS_malloc(_PS_ n);
    if (p)
    {
        SYS_MEMSET(p, 0, n);
    }
    return (FS_VOID *)p;
}

/****************************************************************/
/* try extra hard to reallocate the buffer */
FS_VOID *FSS_realloc(_DS_ FS_VOID *ptr, FS_ULONG n)
{
    FS_BYTE *p;
    int ok = 1;

    if (n >= STATE.server->heap_size)
    {
        STATE.error = ERR_REALLOC_FAIL;
        return 0;
    }
    if (n == 0)
    {
        FSS_free(_PS_ ptr);
        return 0;
    }
    if (ptr == 0)
    {
        return FSS_malloc(_PS_ n);
    }
    while (ok)
    {
        STATE.error = SUCCESS;
        p = (FS_BYTE *)fs__realloc(_PS_ ptr, n);
        if (p)
        {
#ifdef FS_MEM_DBG
            FS_FPRINTF((FS_state_ptr->memdbgfp, "REALLOC %p %p %u %s\n", ptr, p, n,
                        FS_state_ptr->memdbgid ? FS_state_ptr->memdbgid : "[NO_LABEL]"));
            FS_state_ptr->memdbgid = NULL;
            FS_FFLUSH(FS_state_ptr->memdbgfp);
#endif
            return (FS_VOID *)p;
        }
        ok = get_some_back(_PS0_);
    }
    STATE.error = ERR_REALLOC_FAIL;
    return 0;
}

FS_LONG FSS_free(_DS_ FS_VOID *p)
{
    if (p)
    {
#ifdef FS_MEM_DBG
        FS_FPRINTF((FS_state_ptr->memdbgfp, "FREE %p\n", p));
        FS_FFLUSH(FS_state_ptr->memdbgfp);
#endif
        return fs__free(_PS_ p);
    }
    else
    {
        return SUCCESS;
    }
}

#ifndef FS_NO_FS_LUMP
static void *new_thing(_DS_ void **list, FS_ULONG size, FS_ULONG count)
{
    if (size == 0)
    {
        return NULL;
    }
    FS_ALIGN(size);     /* right pad the objects to a "nice" size */

    /* if list empty, we need to add more */
    if (*list == NULL)
    {
        if ((sizeof(FS_MASTER_LUMP) + size) >= STATE.server->heap_size)
        {
            STATE.error = ERR_MALLOC_FAIL;
            return NULL;
        }
        while (count > 0)
        {
            FS_ULONG n;

            n = sizeof(FS_MASTER_LUMP) + (count * size);

#ifdef FS_MEM_DBG
            STATE.memdbgid = "FS_MASTER_LUMP";
#endif
            do
            {
                FS_MASTER_LUMP *master;

                STATE.error = SUCCESS;
                master = fs__malloc(_PS_ n);
                if (master)
                {
                    FS_LUMP *lump;
                    FS_LUMP *ptr;
                    FS_ULONG i;
#ifdef FS_MEM_DBG
                    FS_FPRINTF((FS_state_ptr->memdbgfp, "MALLOC %p %u %s\n", master, n,
                                FS_state_ptr->memdbgid ? FS_state_ptr->memdbgid : "[NO_LABEL]"));
                    FS_state_ptr->memdbgid = NULL;
                    FS_FFLUSH(FS_state_ptr->memdbgfp);
#endif
                    master->ref_count = 0;

                    /* add to list of lumps to be freed by FS_exit() */
                    master->prev = NULL;
                    master->next = STATE.server->list_of_lumps;
                    if (master->next)
                    {
                        master->next->prev = master;
                    }
                    STATE.server->list_of_lumps = master;

                    /* link up the real objects */
                    ptr = lump = (FS_LUMP *)(master + 1);


                    for (i = 0; i < (count - 1); i++)
                    {
                        FS_LUMP *next;

                        next = (FS_LUMP *)((FS_BYTE *)ptr + size);
                        ptr->master = master;
                        ptr->next = next;
                        ptr = next;
                    }
                    ptr->master = master;
                    ptr->next = NULL;

                    /* store the list of new objects */
                    *list = lump;
                }
                else if (!get_some_back(_PS0_))
                {
                    break;
                }
                else if (*list != NULL)
                {
                    STATE.error = SUCCESS;
                }
            }
            while (*list == NULL);

            if (*list != NULL)
            {
                break;
            }
            count >>= 1;
        }
        if (count == 0)
        {
            STATE.error = ERR_MALLOC_FAIL;
            return NULL;
        }
    }
    {
        FS_MASTER_LUMP *master;
        FS_LUMP *lump;

        /* pop the list of new objects */
        lump = (FS_LUMP *)(*list);
        if (lump)
        {
            master = (FS_MASTER_LUMP *)(lump->master);

            master->ref_count++;

            *list = lump->next;

            SYS_MEMSET(lump, 0, size);

            lump->master = master;
        }
        return lump;
    }
}

static void del_thing(_DS_ void **list, void *ptr)
{
#ifdef FS_NO_SORT_FS_LUMP
    /* don't delete it, just prepend it to the list of "new" objects */

    FS_LUMP *lump = ptr;

    lump->next = *list;

    *list = ptr;
#else
    FS_LUMP *lump = ptr;
    FS_MASTER_LUMP *master = (FS_MASTER_LUMP *)(lump->master);

    if (master->ref_count <= 1)
    {
        /* remove everything associated with this master from the list */

        FS_LUMP *prev = NULL;
        FS_LUMP *curr = (FS_LUMP *)(*list);

        while (curr && ((void *)curr < (void *)master))
        {
            prev = curr;
            curr = (FS_LUMP *)(curr->next);
        }
        while (curr && (curr->master == lump->master))
        {
            curr = (FS_LUMP *)(curr->next);
        }
        if (prev)
        {
            prev->next = curr;
        }
        else
        {
            *list = curr;
        }
        /* unlink master */

        if (master->prev)
        {
            master->prev->next = master->next;
        }
        if (master->next)
        {
            master->next->prev = master->prev;
        }
        if ((FS_MASTER_LUMP *)(STATE.server->list_of_lumps) == master)
        {
            STATE.server->list_of_lumps = master->prev;
        }
        FSS_free(_PS_ master);
    }
    else
    {
        /* don't delete it, just insert it into the sorted list of "new" objects */

        FS_LUMP *prev = NULL;
        FS_LUMP *curr = (FS_LUMP *)(*list);

        master->ref_count--;

        while (curr && (lump > curr))
        {
            prev = curr;
            curr = (FS_LUMP *)(curr->next);
        }
        lump->next = curr;

        if (prev)
        {
            prev->next = lump;
        }
        else
        {
            *list = lump;
        }
    }
#endif
}

#define CACHE_ENTRY_BLOCK (sizeof(CACHE_ENTRY)*127) /* SP used (2048) */

CACHE_ENTRY *CACHE_ENTRY_calloc(_DS0_)
{
    return new_thing(_PS_ & STATE.server->CACHE_ENTRY_chain,
                     sizeof(CACHE_ENTRY),
                     CACHE_ENTRY_BLOCK / sizeof(CACHE_ENTRY));
}

void CACHE_ENTRY_free(_DS_ CACHE_ENTRY *c)
{
    del_thing(_PS_ & STATE.server->CACHE_ENTRY_chain, c);
}

/*
 * we could create similar function pairs and "xxx_chain" lists in the server
 * for frequently allocated things which have a life outside of the API call
 * which allocates them ... for instance
 * LFNT, SENV, SFNT, TTF, CFNT, MAPPEDFNT, FNTSET, TABLE_PTR_REC, etc.
 *
 */
#endif
